/*
 * File: XKCDColors.java
 * ==========================================================================
 * A visualizer for the color data gathered by Randall Munroe of xkcd fame.
 * 
 * The data here is a (massively) sanitized version of the raw data from
 * http://xkcd.com/color/colorsurvey.tar.gz.  You can look at this data if you'd
 * like, but be warned that some of the color names are *certainly* NSFW. 
 */
import acm.program.*;
import acm.graphics.*;
import acm.util.*;
import java.util.*;
import java.io.*;

public class XKCDColors extends ConsoleProgram {
	/* The name of the data file holding all of the named RGB triplets. */
	private static final String COLORS_FILE = "xkcd-colors.txt";
	
	/* The display we use to show the colors. */
	private XKCDColorWheel display = new XKCDColorWheel();
	
	/**
	 * Adds the display to the window.
	 */
	public void init() {
		add(display);
	}
	
	/**
	 * Prompts the user for names of colors, then plots them!
	 */
	public void run() {	
		/* Hat guy has stolen the normally witty comment you'd fine here. */
		setFont("DejaVuSerif-BOLD-24");
		
		/* Load the colors data. */
		HashMap<String, ArrayList<int[]>> colors = loadColorsFile();
		display.clear();
		
		while (true) {
			String colorName = readLine("Enter a color: ").toLowerCase();
			
			/* Optionally clear the display. */
			if (colorName.equals("clear")) {
				display.clear();
			}
			/* If we know this color, display all the points matching that color. */
			else if (colors.containsKey(colorName)) {
				ArrayList<int[]> matchingPoints = colors.get(colorName);
				for (int i = 0; i < matchingPoints.size(); i++) {
					display.plotColor(matchingPoints.get(i));
				}
			}
			/* Otherwise, we don't know this color. */
			else {
				println("Sorry, I don't know that color.");
			}
		}
	}
	
	/**
	 * Loads in a table of all of the names of the color data points from
	 * the xkcd color naming set.
	 * 
	 * @return A map from the color names to the list of RGB triplets with
	 *         that name.
	 */
	private HashMap<String, ArrayList<int[]>> loadColorsFile() {
		try {
			/* Open the file for reading. */
			BufferedReader br = new BufferedReader(new FileReader(COLORS_FILE));
			
			/* Construct the HashMap that we will provide back as a result. */
			HashMap<String, ArrayList<int[]>> result = new HashMap<String, ArrayList<int[]>>();
			
			while (true) {
				/* Read the next entry:
				 * 1. The name of the color.
				 * 2. Its red component.
				 * 3. Its green component.
				 * 4. Its blue component.
				 */
				String colorName = br.readLine();
				String r = br.readLine();
				String g = br.readLine();
				String b = br.readLine();
				
				/* If we ran out of data, we're done. */
				if (b == null) break;
				
				/* Construct an array of the colors. */
				int[] color = new int[3];
				color[0] = Integer.parseInt(r);
				color[1] = Integer.parseInt(g);
				color[2] = Integer.parseInt(b);
				
				/* Ensure that there's an ArrayList waiting for us. */
				if (!result.containsKey(colorName)) {
					result.put(colorName, new ArrayList<int[]>());
				}
				
				/* Add this color data to the color list. */
				result.get(colorName).add(color);
			}
			
			br.close();
			return result;			
		} catch (IOException e) {
			throw new ErrorException(e);
		}
	}
}
